package main

import (
	_ "database/driver/mysql"
	"database/orm/gorm"
	"flag"
	"net"
	"net/httpd"
	"os"
	"os/daemon"
	log "os/logger"
	"sync"
	"time"
)

/*
 SELECT * FROM `logs` WHERE `time` > '2015-08-01 00:00:00' AND `time` < '2015-08-02 00:00:00' AND speed > 1
 SELECT * FROM `logs` WHERE `time` > '2015-08-01 00:00:00' AND `time` < '2015-08-02 00:00:00' AND speed >= 0
*/

var (
	server = &GPSServer{
		c: &GPSServerConfig{ // Default config
			cfgfile:   "/etc/gpsd.conf",
			db_host:   "localhost",
			db_port:   3306,
			db_db:     "gpsd",
			db_user:   "root",
			db_pass:   "",
			http_root: "public_html/",
			http_port: 8080,
		},
		clients: make(map[string]*TDeviceInfo),
	}
	html      = NewHTML()
	sidname   = "sid"
	sidexpire = 1 * time.Hour
	logfd     *os.File
)

type (

	// GPS Server - handle client Connections
	GPSServer struct {
		clients  map[string]*TDeviceInfo
		c        *GPSServerConfig // server_config.go
		parselog string
		logfile  string
		db       gorm.DB
		mutex    sync.Mutex
		wait     sync.WaitGroup
		shutdown bool
	}

	// GPS Logger - handle data logging
	EventData struct {
		Active    bool
		AlarmCode int
		AlarmMsg  string
		Time      time.Time
		Latitude  float64
		Longitude float64
		Speed     float64
		Miles     int64
		Angle     float64
		Power     bool // GPS Power
		Acc       bool // Motor Ignition
	}

	// Some Handlers
	LoginHandler func(string) bool
	EventHandler func(string, EventData)

	// Tracking device - GPS
	TDeviceInfo struct {
		Id       string
		IpAddr   string
		LastSeen *time.Time // Last seen online
		adapter  DeviceAdapter
	}

	// SQL database Users
	User struct {
		Id       int32  `sql:"type:int AUTO_INCREMENT",gorm:"primary_key"`
		Name     string `sql:"type:varchar(30)"`
		Password string `sql:"type:varchar(60)"`
	}

	// SQL database Devices
	Device struct {
		Id   int64  `sql:"type:int AUTO_INCREMENT;",gorm:"primary_key"`
		Uid  int64  `sql:"type:int"`
		Imei string `sql:"type:varchar(30)"`
	}

	Presence struct {
		id   int64
		time int64
	}

	// SQL database Logs
	Log struct {
		Id        int64     `sql:"type:int AUTO_INCREMENT;",gorm:"primary_key"`
		Time      time.Time `sql:"DEFAULT:NULL"`
		Did       int64     `sql:"type:int;not null;"`
		Active    bool      `sql:"type:boolean"`
		Latitude  float64   `sql:"type:float(9,6);not null;"`
		Longitude float64   `sql:"type:float(9,6);not null;"`
		Speed     int       `sql:"type:int;not null"`
		Angle     float64   `sql:"type:float(9,6);not null;"`
	}

	Session struct {
		user string
	}
)

func init() {
	log.SetLevel(log.DebugLevel)

	daemon.Setup(&daemon.Config{
		//Uid:         1000,
		//Gid:         1000,
		//Chroot:      "/home/gspd/",
		Pidfile:    "/var/run/gpsd.pid",
		IntHandler: statsHandler,
		//TermHandler: signals,
		//QuitHandler: signals,
		//Usr1Handler: signals,
		//Usr2Handler: signals,
		HupHandler: hupHandler,
	})
	flag.StringVar(&server.parselog, "p", "", "Description")
	flag.Parse()

	if server.parselog == "" {

		err := daemon.Background()
		if err != nil {
			log.Error(err)
		}
		logfd, err := log.OpenLog("/var/log/gpsd.log")
		if err != nil {
			log.Errorf("Open log file error: %s", err)
			return
		}

		log.SetOutput(logfd)

	}
}

// Process SIGHUP Handler
func hupHandler(sig int) bool {
	// Cancel standard behavior
	return true
}

// Process SIGINT Handler
func statsHandler(sig int) bool {
	log.Info("SIGINT received")
	server.Shutdown()
	// Cancel standard behavior
	server.wait.Wait()
	return false
}

func main() {

	err := server.c.LoadConfig()
	if err != nil {
		log.Fatalf("LoadConfig error : %s", err)

	}
	err = server.InitDatabase()
	if err != nil {
		log.Fatalf("Database config error : %s", err)
	}

	if server.parselog != "" {
		err := parse_logfile(server.parselog)
		if err != nil {
			log.Errorf("parse_logfile error: %s", err)
		}
		return
	}

	// Start Listening on Ports
	err = server.ListenTCP(":9090")
	if err != nil {
		log.Fatalf("ListenTCP error : %s", err)
	}

	router, err := gpsd_router()
	if err != nil {
		log.Error(err)
		return
	}

	err = httpd.ListenAndServe(server.c.httpAddr(), router)
	if err != nil {
		log.Error(err)
		return
	}
}

// Server  Listen TCP
func (s *GPSServer) ListenTCP(addr string) error {
	l, err := net.Listen("tcp", addr)
	if err != nil {
		log.Error("Error listening: %s", err)
		return err
	}
	log.Info("GPS service listening on " + addr)
	go func(l net.Listener, addr string) {
		s.wait.Add(1)
		for {
			// Listen for an incoming connection.
			l.(*net.TCPListener).SetDeadline(time.Now().Add(100 * time.Millisecond))
			conn, err := l.Accept()
			// Shutdown request ?
			if server.shutdown == true {
				l.Close()
				log.Infof("Shutdown listener socket %s", addr)
				s.wait.Done()
				return
			}
			if err != nil {
				netErr, ok := err.(net.Error)
				if ok && netErr.Timeout() && netErr.Temporary() {
					continue
				}
				return
			}
			go server.handleTCP(conn)
		}
	}(l, addr)
	return nil
}

// Handle New TCP connection
func (s *GPSServer) handleTCP(c net.Conn) {
	d := &TDeviceInfo{}
	for {

		buf := make([]byte, 1024)
		mlen, err := c.Read(buf)
		if err != nil {
			log.Errorf("Error reading: %s", err)
			return
		}
		rcvd := string(buf)[:mlen]

		if d.Id == "" {
			log.Infof("New Connection from %s", c.RemoteAddr().String())
			d, err = GetDeviceAdapter(rcvd, c)
			if err != nil {
				c.Close()
				log.Errorf("No adapter found for device !")
				return
			}

			// Give ServerHandlers to Device
			d.onLogin(s.onLogin)
			d.onAlarm(s.onAlarm)
			d.onPing(s.onPing)

			// Save to clients list
			s.mutex.Lock()
			s.clients[d.Id] = d
			s.mutex.Unlock()
		}
		// Make sure addapter can Handle message
		err = d.Handle(rcvd)
		if err != nil {
			log.Errorf("Unable to parse message %s", rcvd)
			c.Close()
			return
		}
		//
	}

}

func (s *GPSServer) sqlDevice(device string) *Device {
	devsql := &Device{}
	server.db.Where("imei = ? ", device).First(&devsql)
	if devsql.Id == 0 {
		devsql.Imei = device
		devsql.Uid = 1 // Default
		server.db.Create(&devsql)
		server.db.Where("imei = ? ", device).First(&devsql)
	}

	return devsql
}

// On device Login Request
func (s *GPSServer) onLogin(device string) bool {
	return true
}

// On device Allarm Request
func (s *GPSServer) onAlarm(device string, event EventData) {

}

// Normal GPS Data
func (s *GPSServer) onPing(device string, event EventData) {
	devsql := s.sqlDevice(device)
	if event.Active {
		// Send log to Database
		logdata := &Log{
			Time:      event.Time,
			Did:       devsql.Id,
			Active:    true,
			Latitude:  event.Latitude,
			Longitude: event.Longitude,
			Speed:     int(event.Speed),
			Angle:     event.Angle,
		}
		server.db.Create(&logdata)
	}
}

// Shutdown Server
func (s *GPSServer) Shutdown() {
	log.Info("Shutdown request via SIGTERM")
	s.shutdown = true
	s.wait.Wait()
}

// Initialize Database
func (s *GPSServer) InitDatabase() error {

	db, err := gorm.Open("mysql", s.c.DbString())
	if err != nil {
		return err
	}
	s.db = db
	db.AutoMigrate(&User{}, &Device{}, &Log{})
	// Multiple column index
	//db.Model(&User{}).AddIndex("idx_user_name_age", "name", "age")

	return nil
}
