package daemon

import (
	"fmt"
	"os"
	"os/signal"
	"path/filepath"
	"syscall"
)

type (
	daemon struct {
		name    string   // Proccess name
		uid     uint32   // Switch user to
		gid     uint32   // Switch group to
		chroot  string   // Chroot directory to
		pid     int      // Current proccess pid
		pidfile string   // Pid file location
		args    []string // os.Args
		env     []string // Environment vars
		signal  chan os.Signal

		// Signal Handlers
		hupHandler  SignalHandler // SIGHUP handler
		intHandler  SignalHandler // SIGINT handler
		termHandler SignalHandler // SIGTERM handler
		quitHandler SignalHandler // SIGQUIT handler
		usr1Handler SignalHandler // SIGUSR1 handler
		usr2Handler SignalHandler // SIGUSR2 handler
	}

	Config struct {
		Uid         uint32        // Child uid
		Gid         uint32        // Child gid
		Chroot      string        // Child Chroot
		Pidfile     string        // Child pidfie
		HupHandler  SignalHandler // SIGHUP handler
		IntHandler  SignalHandler // SIGINT handler
		TermHandler SignalHandler // SIGTERM handler
		QuitHandler SignalHandler // SIGQUIT handler
		Usr1Handler SignalHandler // SIGUSR1 handler
		Usr2Handler SignalHandler // SIGUSR2 handler

	}

	// SignalHandler must return true to cancel default signal behavior
	SignalHandler func(int) bool
)

var std = new()

const (
	self_proc = "/proc/self/exe"
	childenv  = "daemon-child-proccess"
	SIGHUP    = 1
	SIGINT    = 2
	SIGQUIT   = 3
	SIGUSR1   = 10
	SIGUSR2   = 12
	SIGTERM   = 15
)

//
// Public Interface
//

// Configure daemon with Settings
func Setup(s *Config) {
	// Setup drop privs
	std.uid = s.Uid
	std.gid = s.Gid
	std.chroot = s.Chroot

	// Setup pidfile
	std.pidfile = s.Pidfile

	// Setup Handlers
	std.hupHandler = s.HupHandler
	std.intHandler = s.IntHandler
	std.termHandler = s.TermHandler
	std.quitHandler = s.QuitHandler
	std.usr1Handler = s.Usr1Handler
	std.usr2Handler = s.Usr2Handler

}

// Run process in background
func Background() error {

	_, err := std.daemon()

	if os.Getenv(childenv) == "" {
		if err != nil {
			return err
		}
		// Parrent exit
		std.Exit(0)
	}
	return nil
}

// Set Enviroment value
func SetEnv(name, value string) {
	std.setEnv(name, value)
}

// Get Enviroment value
func GetEnv(name string) string {
	return std.getEnv(name)
}

// Add Argv to new Proccess
func SetArg(arg string) {
	std.args = append(std.args, arg)
}

// Get current Args
func GetArgs() []string {
	return os.Args
}

// Get current process pid
func GetPid() int {
	return std.pid
}

// Is daemon chrooted ?
func IsChroot() bool {
	return std.chroot != ""
}

// Create new Daemon
func new() *daemon {

	procName := os.Getenv(childenv)

	std := &daemon{
		name:   procName,
		pid:    os.Getpid(),
		args:   os.Args[:1],
		env:    os.Environ(),
		signal: make(chan os.Signal),
	}

	go func(d *daemon) {
		// Add Signal Notifiers
		signal.Notify(d.signal, syscall.SIGTERM, syscall.SIGINT, syscall.SIGQUIT)
		signal.Notify(d.signal, syscall.SIGHUP, syscall.SIGUSR1, syscall.SIGUSR2)

		for {
			select {
			case recv := <-d.signal:
				switch recv {
				case syscall.SIGINT:
					if d.intHandler != nil {
						if d.intHandler(SIGINT) {
							continue
						}
					}
					// Default behavior
					d.Exit(2)

				case syscall.SIGTERM:
					if d.termHandler != nil {
						if d.termHandler(SIGTERM) {
							continue
						}
					}
					// Default behavior
					d.Exit(0)

				case syscall.SIGHUP:
					if d.hupHandler != nil {
						if d.hupHandler(SIGHUP) {
							continue
						}
					}
					// Default behavior
					d.Exit(2)

				case syscall.SIGUSR1:
					if d.usr1Handler != nil {
						d.usr1Handler(SIGUSR1)
					}

				case syscall.SIGUSR2:
					if d.usr2Handler != nil {
						d.usr2Handler(SIGUSR2)
					}

				case syscall.SIGQUIT:
					if d.quitHandler != nil {
						if d.quitHandler(SIGQUIT) {
							continue
						}
					}
					// Default behavior
					d.Exit(2)
				}
			}
		}
	}(std)

	return std
}

// Set Enviroment value
func (d *daemon) setEnv(name, value string) {
	if os.Getenv(childenv) == "" {
		d.env = append(d.env, fmt.Sprintf("%s=%s", name, value))
	} else if name != childenv {
		os.Setenv(name, value)
	}
}

// Return Enviroment value
func (d *daemon) getEnv(name string) string {
	return os.Getenv(name)
}

// New daemon
func (d *daemon) daemon() (int, error) {
	// Code executed by parrent
	if os.Getenv(childenv) == "" {

		// Get process name
		self, err := os.Readlink(self_proc)
		if err != nil {
			return 0, err
		}
		d.name = filepath.Base(self)

		// Setup child env
		d.setEnv(childenv, d.name)

		// Process Attributes
		sysProcAttr := &syscall.SysProcAttr{
			Chroot: d.chroot,
			Setsid: true,
		}

		// Switch User / Group
		if d.uid > 0 || d.gid > 0 {
			sysProcAttr.Credential = &syscall.Credential{
				Uid: d.uid,
				Gid: d.gid,
			}
		}

		// Start Proccess
		proc, err := os.StartProcess(
			d.name,
			d.args,
			&os.ProcAttr{
				Dir: filepath.Dir(self),
				Env: d.env,
				Files: []*os.File{
					nil,
					os.Stdout, // Read Stdout from child (debug)
					os.Stderr, // Read Stderr from child (debug)
				},
				Sys: sysProcAttr,
			},
		)

		if err != nil {
			return 0, err
		}

		return proc.Pid, nil
	}

	// Code executed by child
	if d.pidfile != "" {
		pf, err := os.OpenFile(d.pidfile, os.O_CREATE|os.O_RDWR, os.FileMode(0640))
		if err != nil {
			return d.pid, err
		}
		defer pf.Close()
		_, err = pf.Write([]byte(fmt.Sprintf("%d", d.pid)))
	}
	return d.pid, nil
}

func (d *daemon) Exit(code int) {
	// Make sure we are child process
	if os.Getenv(childenv) != "" && d.pidfile != "" {
		_, err := os.Stat(d.pidfile)
		if err == nil {
			os.Remove(d.pidfile)
		}
	}
	os.Exit(code)
}
