package httpd

import (
	"bufio"
	"compress/gzip"
	"encoding/json"
	"net"
	"net/http"
	"net/url"
	"strings"
	"time"
)

const (
	//---------
	// Gzip Compersion Level
	//---------
	GzipBest          = gzip.BestCompression
	GzipBestSpeed     = gzip.BestSpeed
	GzipDefault       = gzip.DefaultCompression
	GzipNoCompression = gzip.NoCompression
)

//
// Response
//

func (r response) Read() []byte {
	return r.b.Bytes()
}

func (r response) Write(bytes []byte) (int, error) {
	return r.b.Write(bytes)
}

func (r response) Header() http.Header {
	return r.r.Header()
}

func (r response) WriteHeader(status int) {
	if r.hijacked {
		return
	}
	r.r.WriteHeader(status)
}

func (r *response) Hijack() (net.Conn, *bufio.ReadWriter, error) {

	var netConn, rw, err = r.r.(http.Hijacker).Hijack()

	if err != nil {
		return nil, nil, err
	}

	r.hijacked = true
	return netConn, rw, err
}

//
// Context
//
func (c *Context) Param(name string) (value string) {
	for i := range c.params {
		if c.params[i].Key == name {
			return c.params[i].Value
		}
	}
	return ""
}

func (c *Context) UrlParse(u string) (*url.URL, error) {
	return url.Parse(u)
}

// Request Header Contains Token Value
func (c *Context) RequestContainsToken(name, value string) bool {
	for _, v := range c.Request.Header[name] {
		for _, s := range strings.Split(v, ",") {
			if strings.EqualFold(value, strings.TrimSpace(s)) {
				return true
			}
		}
	}
	return false
}

func (c *Context) RequestURL() *url.URL {
	return c.Request.URL
}

// Return RequestURLString
func (c *Context) RequestURLString() string {
	return c.Request.URL.String()
}

func (c *Context) SetCookie(name, value string, expire time.Duration) {
	expiration := time.Now().Add(expire)
	cookie := http.Cookie{
		Name:    name,
		Value:   value,
		Expires: expiration,
	}
	http.SetCookie(c.Response, &cookie)
}

// JSON sends an application/json response with status code.
func (c *Context) JSON(code int, i interface{}) error {
	c.Response.Header().Set(ContentType, ApplicationJSON)
	c.Response.WriteHeader(code)
	return json.NewEncoder(c.Response.b).Encode(i)
}

// String sends a text/plain response with status code.
func (c *Context) String(code int, s string) error {
	c.Response.Header().Set(ContentType, TextPlain)
	c.Response.WriteHeader(code)
	_, err := c.Response.Write([]byte(s))
	return err
}

// HTML sends a text/html response with status code.
func (c *Context) HTML(code int, html string) error {
	c.Response.Header().Set(ContentType, TextHTML)
	c.Response.WriteHeader(code)
	_, err := c.Response.Write([]byte(html))
	return err
}

// NoContent sends a response with no body and a status code.
func (c *Context) NoContent(code int) error {
	c.Response.WriteHeader(code)
	return nil
}

// Error Method
func (c *Context) Error(error string, code int) {
	c.Response.Header().Set(ContentType, "text/plain; charset=utf-8")
	c.Response.WriteHeader(code)
	c.Response.Write([]byte(error))
}

// Redirect Method
func (c *Context) Redirect(url string, code int) {
	c.headerSent = true

	http.Redirect(c.Response, c.Request, url, code)
}

// Send Response
func (c *Context) Send() {

	if c.headerSent || c.Response.hijacked {
		return
	}

	c.headerSent = true

	// Make sure a handler have enabled gzip compression
	if c.Response.Header().Get(ContentEncoding) == "gzip" {
		// Make sure the browser accept Gzip encoding

		if c.RequestContainsToken(AcceptEncoding, "gzip") {
			content := c.Response.Read()
			// Remove previews Content-Length header
			c.Response.Header().Del(ContentLength)
			// If Content-Type header is not set
			if c.Response.Header().Get(ContentType) == "" {
				// Force detection of Content-Type
				c.Response.Header().Set(ContentType, http.DetectContentType(content))
			}

			// Gzip Response
			writer, err := gzip.NewWriterLevel(c.Response.r, c.GzipLevel)
			if err != nil {
				writer, _ = gzip.NewWriterLevel(c.Response.r, GzipBestSpeed)
			}

			writer.Write(content)
			writer.Close()
			return
		}
		// Remove encoding
		c.Response.Header().Del(ContentEncoding)
	}

	c.Response.r.Write(c.Response.Read())
}

func (c *Context) getSubProtocols() []string {
	h := strings.TrimSpace(c.Request.Header.Get(WebsocketProtocol))
	if h == "" {
		return nil
	}
	protocols := strings.Split(h, ",")
	for i := range protocols {
		protocols[i] = strings.TrimSpace(protocols[i])
	}
	return protocols
}

// checkSameOrigin returns true if the origin is not set or is equal to the request host.
func (c *Context) sameOrigin(origin string, host string) bool {
	if len(origin) == 0 {
		return true
	}
	u, err := url.Parse(origin)
	if err != nil {
		return false
	}
	return u.Host == host
}
