package httpd

import (
	"fmt"
	"net/http"
	"strings"
)

// Make sure the Router conforms with the http.Handler interface
var _ http.Handler = NewRouter()

// GET is a shortcut for router.Handle("GET", path, handle)
func (r *Router) Get(path string, handle Handle) error {
	return r.Add("GET", path, handle, nil)
}

// HEAD is a shortcut for router.Handle("HEAD", path, handle)
func (r *Router) Head(path string, handle Handle) error {
	return r.Add("HEAD", path, handle, nil)
}

// OPTIONS is a shortcut for router.Handle("OPTIONS", path, handle)
func (r *Router) Options(path string, handle Handle) error {
	return r.Add("OPTIONS", path, handle, nil)
}

// POST is a shortcut for router.Handle("POST", path, handle)
func (r *Router) Post(path string, handle Handle) error {
	return r.Add("POST", path, handle, nil)
}

// PUT is a shortcut for router.Handle("PUT", path, handle)
func (r *Router) Put(path string, handle Handle) error {
	return r.Add("PUT", path, handle, nil)
}

// PATCH is a shortcut for router.Handle("PATCH", path, handle)
func (r *Router) Patch(path string, handle Handle) error {
	return r.Add("PATCH", path, handle, nil)
}

// DELETE is a shortcut for router.Handle("DELETE", path, handle)
func (r *Router) Delete(path string, handle Handle) error {
	return r.Add("DELETE", path, handle, nil)
}

// Websocket is a shortcut for router.Handle("GET", path, handle)
func (r *Router) Websocket(path string, up Upgrader, handle HandleWs) error {
	return r.Add("GET", path, handle, &up)
}

// Create new SubRouter
func (r *Router) Subrouter(path string) *Router {
	sr := &Router{
		parent:           r,
		trees:            r.trees,
		ErrorLogger:      r.ErrorLogger,
		prefix:           r.subPath(path),
		HandleNotAllowed: false,
	}
	r.children = append(r.children, sr)
	return sr
}

// Handle()
func (r *Router) Add(method, path string, handle interface{}, up *Upgrader) error {

	path = r.subPath(path)

	if path[0] != '/' {
		return fmt.Errorf("path must begin with '/' in path '" + path + "'")
	}

	if r.trees == nil {
		r.trees = make(map[string]*route)
	}
	switch handle.(type) {
	case Handle, HandleWs:
		root := r.trees[method]
		if root == nil {
			root = new(route)
			r.trees[method] = root
		}
		err := root.addRoute(path, handle, up)
		if err != nil {
			return r.Errorf("%s", err)
		}
	default:
		return r.Errorf("Unknown handler !")
	}

	return nil
}

// Before Handler
func (r *Router) BeforeHandle(middleware ...func(Handler) Handler) {
	r.beforeHandle = append(r.beforeHandle, middleware...)
}

// After Handler
func (r *Router) AfterHandle(middleware ...func(Handler) Handler) {
	r.afterHandle = append(r.afterHandle, middleware...)
}

// ServeFiles - http FileServer
func (r *Router) ServeFiles(path string, root string) error {

	if len(path) < 10 || path[len(path)-10:] != "/*filepath" {
		r.Errorf("path must end with /*filepath in path '" + path + "'")
	}

	fileServer := http.FileServer(http.Dir(root))

	r.Add("GET", path, Handle(
		func(c *Context) {
			c.Request.URL.Path = c.Param("filepath")
			fileServer.ServeHTTP(c.Response, c.Request)
		},
	), nil)
	return nil
}

// get SubPath of route
func (r *Router) subPath(p string) string {
	pre := r.prefix

	if (pre == "/" || pre[:len(pre)-1] == "/") && p[:1] == "/" {
		pre = pre[:len(pre)-1]
	}

	return pre + p
}

// ServeHTTP
func (r *Router) ServeHTTP(res http.ResponseWriter, req *http.Request) {

	child := r.getLastChild(req.URL.Path, r)
	hndlr := newRouter()

	for i := len(child.afterHandle) - 1; i >= 0; i-- {
		hndlr = child.afterHandle[i](hndlr)
	}

	hndlr = func(next Handler) Handler {
		return HandlerFunc(func(c *Context) {
			child.Handle(c)
			next.Handle(c)
		})
	}(hndlr)

	for i := len(child.beforeHandle) - 1; i >= 0; i-- {
		hndlr = child.beforeHandle[i](hndlr)
	}

	context := newContext(req, res)

	hndlr.Handle(context)
}

// Handle
func (r *Router) Handle(c *Context) {

	if r.PanicHandler != nil {
		defer r.recv(c)
	}
	if root := r.trees[c.RequestMethod]; root != nil {
		path := c.Request.URL.Path

		if handle, ps, tsr, up := root.getValue(path); handle != nil {
			c.params = ps
			switch handle.(type) {

			case Handle:
				handle.(Handle)(c)

			case HandleWs:
				ws, err := r.websocket(c, up)
				if err != nil {
					r.Errorf("%s", err)
					return
				}
				handle.(HandleWs)(c, ws)
			}
			return
		} else if c.RequestMethod != "CONNECT" && path != "/" {
			code := 301
			if c.RequestMethod != "GET" {
				code = 307
			}

			if tsr {
				if len(path) > 1 && path[len(path)-1] == '/' {
					c.Request.URL.Path = path[:len(path)-1]
				} else {
					c.Request.URL.Path = path + "/"
				}
				c.Redirect(c.RequestURLString(), code)
				return
			}

			fixedPath, found := root.findCaseInsensitivePath(
				cleanPath(path),
				true,
			)
			if found {
				c.Request.URL.Path = string(fixedPath)
				c.Redirect(c.RequestURLString(), code)
				return
			}

		}
	}

	// Handle 405
	if r.HandleNotAllowed {
		for method := range r.trees {
			if method == c.RequestMethod {
				continue
			}
			handle, _, _, _ := r.trees[method].getValue(c.Request.URL.Path)
			if handle != nil {
				if r.MethodNotAllowed != nil {
					r.MethodNotAllowed(c)
				} else {
					c.Error(
						StatusText(StatusMethodNotAllowed),
						StatusMethodNotAllowed,
					)
				}
				return
			}
		}
	}

	if r.MethodNotFound != nil {
		r.MethodNotFound(c)
	} else {
		http.NotFound(c.Response, c.Request)
	}
}

// Handle Websocket
func (r *Router) websocket(c *Context, up Upgrader) (*Conn, error) {
	if !c.RequestContainsToken(WebsocketVersion, "13") {
		c.Error(StatusText(StatusBadRequest), StatusBadRequest)
		return nil, r.Errorf("Websocket: version != 13")
	}

	if !c.RequestContainsToken(Connection, "upgrade") {
		c.Error(StatusText(StatusBadRequest), StatusBadRequest)
		return nil, r.Errorf("Websocket: missing header Connection:upgrade")
	}

	if !c.RequestContainsToken(Upgrade, "websocket") {
		c.Error(StatusText(StatusBadRequest), StatusBadRequest)
		return nil, r.Errorf("Websocket: missing header: Upgrade:websocket")
	}

	checkOrigin := up.CheckOrigin
	if checkOrigin == nil {
		checkOrigin = c.sameOrigin
	}

	//
	u, err := c.UrlParse(c.Request.Header.Get(Origin))

	if err != nil {
		c.Error(StatusText(StatusForbidden), StatusForbidden)
		return nil, r.Errorf("Websocket: cannot parse origin")
	}

	if !checkOrigin(u.Host, c.RequestURL().Host) {
		c.Error(StatusText(StatusForbidden), StatusForbidden)
		return nil, r.Errorf("Websocket: invalid origin")
	}

	challenge := c.Request.Header.Get(WebsocketKey)
	if challenge == "" {
		c.Error(StatusText(StatusBadRequest), StatusBadRequest)
		return nil, r.Errorf("Websocket: missing header: Sec-Websocket-Key")
	}

	// WebsocketProtocol negocitation
	subprotocol := ""
	rsubprotocol := c.getSubProtocols()
	if len(up.Subprotocols) > 0 && len(rsubprotocol) > 0 {
	Scan:
		for _, sproto := range up.Subprotocols {
			for _, cproto := range rsubprotocol {
				if cproto == sproto {
					subprotocol = sproto
					break Scan
				}
			}
		}

		if subprotocol == "" {
			c.Error(StatusText(StatusBadRequest), StatusBadRequest)
			return nil, r.Errorf("Websocket: server does not implement %s", rsubprotocol)
		}
	}

	net, buf, err := c.Response.Hijack()
	if err != nil {

		c.Error(StatusText(StatusBadRequest), StatusBadRequest)
		return nil, r.Errorf("Websocket: unable to hijack connection")
	}

	return up.upgrade(challenge, subprotocol, net, buf)

}

func (r *Router) getLastChild(path string, last *Router) *Router {
	for _, child := range last.children {
		if strings.HasPrefix(path, child.prefix) {
			last = child.getLastChild(path, child)
		}
	}
	return last
}

func (r *Router) Errorf(err string, a ...interface{}) error {
	r.ErrorLogger.Printf("httpd.Router: "+err, a...)
	return fmt.Errorf(err, a...)
}

func (r *Router) recv(c *Context) {
	if rcv := recover(); rcv != nil {
		//r.PanicHandler(c, rcv)
	}
}
