package main

import (
	"net"
	log "os/logger"
	"strings"
)

type (
	//
	// Interface for GPS adapters
	//
	DeviceAdapter interface {
		Handle(string) error
		Send(string, string)
		OnLogin(LoginHandler)
		OnAlarm(EventHandler)
		OnPing(EventHandler)
	}

	// Implement DeviceAdapter interface
	TK103 struct {
		DeviceId string
		c        net.Conn
		onLogin  LoginHandler
		onAlarm  EventHandler
		onPing   EventHandler
	}
)

// Handles incoming requests.
func (a *TK103) Handle(rcvd string) error {

	cmd_start := strings.Index(rcvd, "B")
	if cmd_start > 13 {
		log.Error("DeviceId too long")
		return nil
	}
	a.DeviceId = rcvd[1:cmd_start]
	cmd := rcvd[cmd_start : cmd_start+4]

	switch cmd {

	case "BP05": // Login Request
		if a.onLogin != nil {
			if a.onLogin(a.DeviceId) == true {
				a.Send("AP05", "")
			}
			return nil
		}
		// Auto accept login
		a.Send("AP05", "")

	case "BP00": // Handshake Request
		a.Send("AP01", "HSO")

	case "BR00": // Ping Request
		res := a._parse(rcvd[cmd_start+4 : len(rcvd)-1])
		if a.onPing != nil {
			a.onPing(a.DeviceId, *res)
		}

	case "BO01": // Alarm Request
		alrm := rcvd[cmd_start+4 : cmd_start+5]
		res := a._parse(rcvd[cmd_start+5 : len(rcvd)-1])

		code := -1
		message := ""
		switch alrm {
		case "0":
			code = 0
			message = "Vehicle Power Off"
		case "1":
			code = 1
			message = "The vehicle suffers an acciden"
		case "2":
			code = 2
			message = "Driver sends a S.O.S."
		case "3":
			code = 3
			message = "The alarm of the vehicle is activated"
		case "4":
			code = 4
			message = "Vehicle is below the min speed setted"
		case "5":
			code = 5
			message = "Vehicle is over the max speed setted"
		case "6":
			code = 6
			message = "Out of geo fence"
		}

		res.AlarmCode = code
		res.AlarmMsg = message

		if a.onAlarm != nil {
			a.onAlarm(a.DeviceId, *res)
		}
		a.Send("AS01", alrm)

	default:
		log.Infof("Unknown message from %s", a.DeviceId)
		log.Printf("Message :%s", rcvd)
	}
	return nil
}

// Setup Login Handler
func (a *TK103) OnLogin(handler LoginHandler) {
	a.onLogin = handler
}

// Setup Alarm Hanlder
func (a *TK103) OnAlarm(handler EventHandler) {
	a.onAlarm = handler
}

// Setup Ping Handler
func (a *TK103) OnPing(handler EventHandler) {
	a.onPing = handler
}

func (tk *TK103) _parse(data string) *EventData {

	yy := data[0:2]         // YYMMDD - Year
	mm := data[2:4]         // YYMMDD - Month
	dd := data[4:6]         // YYMMDD - Month
	active := data[6:7]     // The availability of GPS data
	latstr := data[7:9]     // Latitude - Minute
	latmstr := data[9:16]   // latitude - Seconds
	latind := data[16:17]   // Latitude indicator "N" or "S"
	longstr := data[17:20]  // Longitue - Minute
	longmstr := data[20:27] // Longitude - Seconds
	longind := data[27:28]  // Longture indicator "E" or "V"
	speedstr := data[28:33] // The unit is km/h
	timeh := data[33:35]    //HHMMSS - hours
	timem := data[35:37]    //HHMMSS - minutes
	times := data[37:39]    //HHMMSS - seconds
	anglestr := data[39:44] // Orientation
	//io := data[44:52] // IO State
	//mil := data[52:53] // Milepost
	//milstr := data[53:61] // Km

	// Process Latitude
	latitude, err := ToDecimalDegrees(latstr, latmstr)
	if err != nil {
		log.Errorf("Val: %s'%s,Err: %s", latstr, latmstr, err)
	}
	if latind == "S" {
		latitude = -latitude
	}

	// Process Longitude
	longitude, err := ToDecimalDegrees(longstr, longmstr)
	if err != nil {
		log.Errorf("Val: %s'%s,Err: %s", longstr, longmstr, err)
	}
	if longind == "W" {
		longitude = -longitude
	}

	// Process Speed
	speed, err := ToDecimal(speedstr)
	if err != nil {
		log.Errorf("Val: %s,Err: %s", speedstr, err)
	}

	// Process Time
	time, err := FormatTimeShort(yy, mm, dd, timeh, timem, times)
	if err != nil {
		log.Errorf("Val: 20%s/%s/%s %s:%s:%s, Err: %s",
			yy, mm, dd, timeh, timem, times, err)
	}

	// Process Orientation
	angle, err := ToDecimal(anglestr)
	if err != nil {
		log.Errorf("Val: %s,Err: %s", anglestr, err)
	}

	return &EventData{
		Active:    (active == "A"),
		Time:      time,
		Latitude:  latitude,
		Longitude: longitude,
		Speed:     speed,
		Angle:     angle,
	}
}

func (tk *TK103) Send(cmd, data string) {
	message := "(" + strings.Join([]string{tk.DeviceId, cmd, data}, "") + ")"
	_, err := tk.c.Write([]byte(message))
	if err != nil {
		log.Error("Failed to send cmd " + cmd + " to " + tk.DeviceId)
	}
}
