package session

import (
	"crypto/sha1"
	"fmt"
	"sync"
	"time"
)

type (
	session struct {
		sessions map[string]*Session
		m        sync.Mutex
		Expire   time.Duration
	}

	Session struct {
		sid    string
		token  string
		Data   interface{}
		expire *time.Time
	}
)

const (
	sGarbage = time.Duration(10 * time.Minute)
	sExpire  = time.Duration(1 * time.Hour)
)

var (
	std = &session{
		sessions: make(map[string]*Session),
		Expire:   sExpire,
	}
)

func init() {
	time.AfterFunc(sGarbage, std.sessiongc)
}

// Set Global Session Expire Duration
func SetExpire(expire int64) {
	std.m.Lock()
	std.Expire = time.Duration(expire) * time.Second
	std.m.Unlock()
}

// Get Session Data
func Get(name string) *Session {

	if name == "" {
		return nil
	}

	std.m.Lock()
	ses, found := std.sessions[name]
	std.m.Unlock()

	// New session or session expired
	if !found {
		return nil
	} else if ses.expired() {
		delete(std.sessions, name)
		return nil
	}

	return ses
}

// Get Session Id
func (s *Session) Id() string {
	return s.sid
}

// Get CSRF token
func (s *Session) Token() string {
	if s.token == "" {
		s.token = newRandomId()
	}
	return s.token
}

// Validate Token
func (s *Session) Validate(token string) bool {
	if token == s.token {
		s.token = newRandomId()
		return true
	}
	return false
}

// Set Session Data
func Set(data *Session) {
	std.m.Lock()
	expire := time.Now().Add(std.Expire)
	data.expire = &expire
	std.sessions[data.sid] = data
	std.m.Unlock()
}

// Create new Session
func New() *Session {
	expire := time.Now().Add(std.Expire)
	return &Session{
		sid:    newRandomId(),
		expire: &expire,
	}
}

// Check if session is expired
func (s *Session) expired() bool {
	if s.expire == nil {
		return false
	}
	return s.expire.Before(time.Now())
}

// Garbage Collector
func (s *session) sessiongc() {
	s.m.Lock()
	for _, ses := range s.sessions {
		if ses.expired() {
			delete(s.sessions, ses.Id())
		}
	}
	s.m.Unlock()
	time.AfterFunc(sGarbage, s.sessiongc)
}

// New Random Id
func newRandomId() string {
	h := sha1.New()
	c := []byte(time.Now().String())
	h.Write(c)
	return fmt.Sprintf("%x", h.Sum(nil))

}
