package httpd

import (
	"bufio"
	"bytes"
	"errors"
	"log"
	"net"
	"net/http"
	"os"
	"time"
)

const (

	//-------------
	// Media types
	//-------------

	ApplicationJSON     = "application/json"
	ApplicationProtobuf = "application/protobuf"
	ApplicationMsgpack  = "application/msgpack"
	TextPlain           = "text/plain"
	TextHTML            = "text/html"
	ApplicationForm     = "application/x-www-form-urlencoded"
	MultipartForm       = "multipart/form-data"

	//---------
	// Headers
	//---------

	Accept             = "Accept"
	AcceptEncoding     = "Accept-Encoding"
	ContentDisposition = "Content-Disposition"
	ContentEncoding    = "Content-Encoding"
	ContentLength      = "Content-Length"
	ContentType        = "Content-Type"
	Authorization      = "Authorization"

	// Websocket related
	Connection          = "Connection"
	Upgrade             = "Upgrade"
	Origin              = "Origin"
	WebsocketKey        = "Sec-Websocket-Key"
	WebsocketProtocol   = "Sec-Websocket-Protocol"
	WebsocketVersion    = "Sec-Websocket-Version"
	WebSocketExtensions = "Sec-WebSocket-Extensions"

	//---------
	// Status Codes
	//---------
	StatusAccepted                     = http.StatusAccepted
	StatusBadGateway                   = http.StatusBadGateway
	StatusBadRequest                   = http.StatusBadRequest
	StatusConflict                     = http.StatusConflict
	StatusContinue                     = http.StatusContinue
	StatusCreated                      = http.StatusCreated
	StatusExpectationFailed            = http.StatusExpectationFailed
	StatusForbidden                    = http.StatusForbidden
	StatusFound                        = http.StatusFound
	StatusGatewayTimeout               = http.StatusGatewayTimeout
	StatusGone                         = http.StatusGone
	StatusHTTPVersionNotSupported      = http.StatusHTTPVersionNotSupported
	StatusInternalServerError          = http.StatusInternalServerError
	StatusLengthRequired               = http.StatusLengthRequired
	StatusMethodNotAllowed             = http.StatusMethodNotAllowed
	StatusMovedPermanently             = http.StatusMovedPermanently
	StatusMultipleChoices              = http.StatusMultipleChoices
	StatusNoContent                    = http.StatusNoContent
	StatusNonAuthoritativeInfo         = http.StatusNonAuthoritativeInfo
	StatusNotAcceptable                = http.StatusNotAcceptable
	StatusNotFound                     = http.StatusNotFound
	StatusNotImplemented               = http.StatusNotImplemented
	StatusNotModified                  = http.StatusNotModified
	StatusOK                           = http.StatusOK
	StatusPartialContent               = http.StatusPartialContent
	StatusPaymentRequired              = http.StatusPaymentRequired
	StatusPreconditionFailed           = http.StatusPreconditionFailed
	StatusProxyAuthRequired            = http.StatusProxyAuthRequired
	StatusRequestEntityTooLarge        = http.StatusRequestEntityTooLarge
	StatusRequestTimeout               = http.StatusRequestTimeout
	StatusRequestURITooLong            = http.StatusRequestURITooLong
	StatusRequestedRangeNotSatisfiable = http.StatusRequestedRangeNotSatisfiable
	StatusResetContent                 = http.StatusResetContent
	StatusSeeOther                     = http.StatusSeeOther
	StatusServiceUnavailable           = http.StatusServiceUnavailable
	StatusSwitchingProtocols           = http.StatusSwitchingProtocols
	StatusTeapot                       = http.StatusTeapot
	StatusTemporaryRedirect            = http.StatusTemporaryRedirect
	StatusUnauthorized                 = http.StatusUnauthorized
	StatusUnsupportedMediaType         = http.StatusUnsupportedMediaType
	StatusUseProxy                     = http.StatusUseProxy
)

type (

	// Handler context
	Context struct {
		headerSent    bool
		challenge     string
		Request       *http.Request
		Response      response
		GzipLevel     int
		RemoteAddr    string
		RequestURI    string
		RequestMethod string
		Session       interface{}
		params        []routeParam
	}

	Handle func(*Context)

	HandleWs func(*Context, *Conn)

	// Handler
	Handler interface {
		Handle(*Context)
	}

	HandlerFunc func(*Context)

	Router struct {
		prefix           string
		parent           *Router
		children         []*Router
		trees            map[string]*route
		ErrorLogger      *log.Logger
		MethodNotFound   HandlerFunc
		HandleNotAllowed bool
		MethodNotAllowed HandlerFunc
		PanicHandler     func(Context, interface{})
		beforeHandle     []func(Handler) Handler
		afterHandle      []func(Handler) Handler
	}

	Upgrader struct {
		HandshakeTimeout time.Duration
		ReadBufferSize   int
		WriteBufferSize  int
		Subprotocols     []string
		CheckOrigin      func(string, string) bool
	}

	Conn struct {
		conn        net.Conn
		isServer    bool
		subprotocol string

		// Write fields
		mu        chan bool // used as mutex to protect write to conn and closeSent
		closeSent bool      // true if close message was sent

		// Message writer fields.
		writeErr       error
		writeBuf       []byte // frame is constructed in this buffer.
		writePos       int    // end of data in writeBuf.
		writeFrameType int    // type of the current frame.
		writeSeq       int    // incremented to invalidate message writers.
		writeDeadline  time.Time

		// Read fields
		readErr       error
		br            *bufio.Reader
		readRemaining int64 // bytes remaining in current frame.
		readFinal     bool  // true the current message has more frames.
		readSeq       int   // incremented to invalidate message readers.
		readLength    int64 // Message size.
		readLimit     int64 // Maximum message size.
		readMaskPos   int
		readMaskKey   [4]byte
		handlePong    func(string) error
		handlePing    func(string) error
	}

	// route
	route struct {
		path      string
		wildChild bool
		nType     nodeType
		maxParams uint8
		indices   string
		children  []*route
		upgrader  Upgrader
		handle    interface{}
		priority  uint32
	}

	// Private structs
	routeParam struct {
		Key   string
		Value string
	}

	response struct {
		hijacked bool
		r        http.ResponseWriter
		b        *bytes.Buffer
	}
	finalRouter struct{}
)

// Convert ServeHTTP to Handle
func (f HandlerFunc) Handle(c *Context) {
	f(c)
}

//
func NewRouter() *Router {
	return &Router{
		prefix:           "/",
		ErrorLogger:      log.New(os.Stderr, "", log.LstdFlags),
		HandleNotAllowed: false,
	}
}

func ListenAndServe(addr string, handler http.Handler) error {
	return http.ListenAndServe(addr, handler)
}

func ListenAndServeTLS(addr string, certFile string, keyFile string, handler http.Handler) error {
	return http.ListenAndServeTLS(addr, certFile, keyFile, handler)
}

func StatusText(code int) string {
	return http.StatusText(code)
}

//
// Private functions
//

// Create new Context
func newContext(request *http.Request, responsew http.ResponseWriter) *Context {
	return &Context{
		Request: request,
		Response: response{
			hijacked: false,
			r:        responsew,
			b:        &bytes.Buffer{},
		},

		RemoteAddr:    request.RemoteAddr,
		RequestURI:    request.RequestURI,
		RequestMethod: request.Method,
		GzipLevel:     GzipBestSpeed,
		params:        nil,
	}
}

// Create new Final Router
func newRouter() Handler {
	return Handler(&finalRouter{})
}

//  Final router Handle function
func (r *finalRouter) Handle(c *Context) {
	c.Send()
}

func (u *Upgrader) upgrade(challenge string, subprotocol string, netConn net.Conn, rw *bufio.ReadWriter) (*Conn, error) {

	var br *bufio.Reader
	br = rw.Reader

	if br.Buffered() > 0 {
		netConn.Close()
		return nil, errors.New("websocket: client sent data before handshake is complete")
	}

	con := newConn(netConn, true, u.ReadBufferSize, u.WriteBufferSize)

	con.subprotocol = subprotocol

	p := con.writeBuf[:0]
	p = append(p, "HTTP/1.1 101 Switching Protocols\r\nUpgrade: websocket\r\nConnection: Upgrade\r\nSec-WebSocket-Accept: "...)
	p = append(p, computeAcceptKey(challenge)...)
	p = append(p, "\r\n"...)
	if con.subprotocol != "" {
		p = append(p, "Sec-Websocket-Protocol: "...)
		p = append(p, con.subprotocol...)
		p = append(p, "\r\n"...)
	}

	p = append(p, "\r\n"...)

	// Clear deadlines set by HTTP server.
	netConn.SetDeadline(time.Time{})

	if u.HandshakeTimeout > 0 {
		netConn.SetWriteDeadline(time.Now().Add(u.HandshakeTimeout))
	}

	if _, err := netConn.Write(p); err != nil {
		netConn.Close()
		return nil, err
	}

	if u.HandshakeTimeout > 0 {
		netConn.SetWriteDeadline(time.Time{})
	}

	return con, nil
}
