package cluster

import (
	"fmt"
	"math/rand"
	"net"
	"time"
)

// Public Peer struct
type Peer struct {
	Id      int64
	Name    string
	Domain  string
	TCPAddr *net.TCPAddr
}

// Peer id to string
func (p *Peer) StringID() string {
	return fmt.Sprintf("%d", p.Id)
}

// Peer address to string
func (p *Peer) StringAddr() string {
	return p.TCPAddr.String()
}

// Comapre peer address
func (p *Peer) equal(p2 *Peer) bool {
	return p.TCPAddr.IP.Equal(p2.TCPAddr.IP) &&
		p.TCPAddr.Port == p2.TCPAddr.Port &&
		p.TCPAddr.Zone == p2.TCPAddr.Zone
}

// Generate a new Peer
func NewPeer(name string, domain string, laddr string) (*Peer, error) {
	var err error
	rand.Seed(time.Now().UTC().UnixNano())

	peer := &Peer{
		Id:     rand.Int63(),
		Name:   name,
		Domain: domain,
	}

	if peer.TCPAddr, err = net.ResolveTCPAddr("tcp", laddr); err != nil {
		return nil, err
	}
	return peer, nil
}

// Accept connection routine
func (c *Cluster) acceptRoutine() {
	defer c.tcp.Close()
	for {
		conn, err := c.tcp.Accept()
		if err != nil {
			// Connection closed , exit this routine
			return
		}

		// Receive and decode routine
		go func(c *Cluster, conn net.Conn) {

			dec := newDecoder(conn)

			c.optlock.RLock()
			networkTimeout := time.Duration(c.networkTimeout) * time.Second
			c.optlock.RUnlock()

			deadline := time.Now().Add(networkTimeout)
			if err := conn.SetReadDeadline(deadline); err != nil {
				c.warn("Timeout receiving from '(%s, %s)': %s", conn.LocalAddr(), conn.RemoteAddr(), err)
				return
			}

			msg, err := dec.Decode()
			if err != nil {
				c.err("Could decode message from '(%s, %s)': %s", conn.LocalAddr(), conn.RemoteAddr(), err)
				conn.Close()
				return
			}

			remoteaddr, err := net.ResolveTCPAddr("tcp", conn.RemoteAddr().String())
			if err != nil {
				c.err("Error resolving remote host address %s, %s", remoteaddr, err)
				return
			}
			msg.remote = remoteaddr

			c.message <- msg
		}(c, conn)

	}
}

// Send Heartbeats routing
func (c *Cluster) heartbeatRoutine() {
	c.waitgroup.Add(1)
	defer c.waitgroup.Done()

	check := func() {
		c.peerslock.RLock()
		defer c.peerslock.RUnlock()

		for _, peer := range c.peers {
			lastheard, ok := c.heartbeats[peer.Id]
			if !ok {
				c.debug("Never heard from '%s'", peer.StringID())
				go c.Connect(peer.TCPAddr.String())
				continue
			} else if time.Since(lastheard) > c.getHealthyInterval()*2 {
				c.warn("Remote '%s' hasn't responded in %s.",
					peer.StringID(),
					c.getHealthyInterval()*2,
				)

				// Do an ungraceful disconnect since we might want to try
				// reconnecting later.
				go c.onNodeExit(peer, false)
				continue
			}

			if err := c.Send(peer, MsgHeartbeat, nil); err != nil {
				continue
			}
		}
	}
	for {
		select {
		case <-c.heartbeatQuit:
			c.debug("Shutting down hearbeat ...")
			return
		case <-time.After(c.getHealthyInterval()):
			check()
		}
	}
}

// Receive messageRouting
func (c *Cluster) receiveRoutine() {
	c.waitgroup.Add(1)
	defer c.waitgroup.Done()

	for {
		select {

		case <-c.receiveQuit:
			c.debug("Shutting down receive thread ...")
			return

		case msg := <-c.message:
			c.peerslock.Lock()
			c.heartbeats[msg.From] = time.Now()
			c.peerslock.Unlock()

			switch msg.Type {
			// Node joins the cluster
			case MsgJoin:
				var peer Peer
				err := msg.Decode(&peer)

				if err != nil {
					c.warn("Unable to decode Join message from %s",
						msg.remote.IP)
					return
				}

				// Rewrite remote Addr with what we see in connection
				if peer.TCPAddr.IP.String() == "0.0.0.0" {
					peer.TCPAddr.IP = msg.remote.IP
				}

				c.debug("Join received from %s@%s ",
					peer.StringID(),
					peer.StringAddr(),
				)

				c.onNodeJoin(peer)

			// Node
			case MsgPeers:
				var peerlist []Peer
				err := msg.Decode(&peerlist)

				if err != nil {
					c.warn("Unable to decode Peers message from %s",
						msg.remote.IP)
					return
				}

				if rpeer, known := c.GetPeerById(msg.From); known == true {
					c.debug("Peers list received from peer %s@%s ",
						rpeer.StringID(),
						rpeer.StringAddr(),
					)

					for _, peer := range peerlist {
						// Lear learn the remote ip from connection
						if peer.Id != c.self.Id {
							c.debug("Learning new peer %s@%s from %s@%s",
								peer.StringID(),
								peer.StringAddr(),
								rpeer.StringID(),
								rpeer.StringAddr(),
							)
							c.onNodeJoin(peer)
						}
					}
				} else {
					c.warn("Peers list received unknown peer %s ",
						msg.remote.IP.String(),
					)
				}

			case MsgHeartbeat:
				if peer, known := c.GetPeerById(msg.From); known == true {
					c.debug("Heartbeat message received from %s@%s",
						peer.StringID(),
						peer.StringAddr(),
					)
					// Send Heartbeat notification
					for _, app := range c.applications {
						app.OnHeartbeat(peer)
					}
				}

			// Node Leaves the cluster
			case MsgLeave:
				if peer, known := c.GetPeerById(msg.From); known == true {
					c.debug("Remove message from %s@%s",
						peer.StringID(),
						peer.StringAddr(),
					)
					c.onNodeExit(peer, true)
				}

			case MsgBroadcast:
				if peer, known := c.GetPeerById(msg.From); known == true {
					c.debug("Broadcast message received from %s@%s",
						peer.StringID(),
						peer.StringAddr(),
					)
					for _, app := range c.applications {
						app.OnBroadcast(
							&BroadcastMessage{
								From:    peer,
								Payload: msg.Payload,
							})
					}
				}

			default:

				if peer, known := c.GetPeerById(msg.From); known == true {
					c.debug("General message received from %s@%s",
						peer.StringID(),
						peer.StringAddr(),
					)
					for _, app := range c.applications {
						app.OnMessage(
							&Message{
								From:    peer,
								Type:    msg.Type,
								Payload: msg.Payload,
							})
					}
				}
			}
		}
	}
}

// Return a list of peers from same domain
func (c *Cluster) getPeersByDomain(domain string) []Peer {
	c.peerslock.RLock()
	defer c.peerslock.RUnlock()

	var peers []Peer
	for _, r := range c.peers {
		// Send only list of peers from same domain
		if r.Domain == domain {
			peers = append(peers, r)
		}
	}
	return peers
}

//  getHeartbeatInterval
func (c *Cluster) getHealthyInterval() time.Duration {
	c.optlock.RLock()
	defer c.optlock.RUnlock()

	return time.Duration(c.heartbeatFrequency) * time.Second
}

// Log debug level message
func (c *Cluster) debug(format string, v ...interface{}) {
	c.optlock.RLock()
	defer c.optlock.RUnlock()

	if c.logLevel <= LogLevelDebug {
		c.log.Printf("%s[%d] <debug> %s",
			c.self.Name,
			c.self.Id,
			fmt.Sprintf(format, v...),
		)
	}
}

// Log warn level message
func (c *Cluster) warn(format string, v ...interface{}) {
	c.optlock.RLock()
	defer c.optlock.RUnlock()

	if c.logLevel <= LogLevelWarn {
		c.log.Printf("%s[%d] <warning> %s",
			c.self.Name,
			c.self.Id,
			fmt.Sprintf(format, v...),
		)

	}
}

// Log error level message
func (c *Cluster) err(format string, v ...interface{}) {
	c.optlock.RLock()
	defer c.optlock.RUnlock()

	if c.logLevel <= LogLevelError {
		c.log.Printf("%s[%d] <error> %s",
			c.self.Name,
			c.self.Id,
			fmt.Sprintf(format, v...),
		)
	}
}
