package cluster

import (
	"bytes"
	"encoding/gob"
	"io"
	"net"
	"time"
)

// Public message struct
type Message struct {
	From    Peer
	Type    int8
	Payload []byte
}

// Public broadcast message struct
type BroadcastMessage struct {
	From    Peer
	Payload []byte
}

// Internal message structure
type message struct {
	Id      int64 // Message            ID
	From    int64 // Source        Peer.Id
	To      int64 // Destintaion   Peer.Id
	Type    int8  // Message type
	Payload []byte
	remote  *net.TCPAddr
}

// message Encoder
type messageEncoder struct {
	msg *message
	enc *gob.Encoder
}

// message Decoder
type messageDecoder struct {
	msg *message
	dec *gob.Decoder
}

// Decode message.Payload into given interface
func (m *message) Decode(v interface{}) error {
	b := bytes.NewReader(m.Payload)
	r := gob.NewDecoder(b)
	if err := r.Decode(v); err != nil {
		return err
	}
	return nil
}

// Create a new Decoder instance
func newDecoder(reader io.Reader) *messageDecoder {
	dec := &messageDecoder{
		msg: new(message),
		dec: gob.NewDecoder(reader),
	}
	return dec
}

// Decode received message
func (m *messageDecoder) Decode() (*message, error) {
	err := m.dec.Decode(m.msg)
	if err != nil {
		return nil, err
	}
	return m.msg, nil
}

// Create new Encoder instance
func newEncoder(writer io.Writer) *messageEncoder {
	enc := &messageEncoder{
		msg: new(message),
		enc: gob.NewEncoder(writer),
	}
	return enc
}

// Encode Message
func (m *messageEncoder) Encode(from int64, to int64, mtype int8, data interface{}) error {
	var payload []byte

	switch d := data.(type) {

	case []byte:
		payload = d
	case string:
		payload = []byte(d)
	case nil:

	default:
		b := new(bytes.Buffer)
		w := gob.NewEncoder(b)
		if err := w.Encode(data); err != nil {
			return err
		}
		payload = b.Bytes()
	}

	m.msg = &message{
		Id:      time.Now().UnixNano(),
		From:    from,
		To:      to,
		Type:    mtype,
		Payload: payload,
	}

	return m.enc.Encode(m.msg)
}
