package cluster

import (
	"crypto/tls"
	"fmt"
	"log"
	"os"
	"sync"
	"testing"
	"time"
)

var rsaCertPEM = `-----BEGIN CERTIFICATE-----
MIIDxzCCAq+gAwIBAgIJAPWnIg4IdFtuMA0GCSqGSIb3DQEBCwUAMHoxCzAJBgNV
BAYTAlJPMRAwDgYDVQQIDAdSb21hbmlhMQ8wDQYDVQQHDAZHYWxhdGkxFTATBgNV
BAoMDENsdXN0ZXItVGVzdDEQMA4GA1UEAwwHY2x1c3RlcjEfMB0GCSqGSIb3DQEJ
ARYQdGVjaEBjbHVzdGVyLm5ldDAeFw0xNDA2MDYxNDEyMjlaFw0xNTA2MDYxNDEy
MjlaMHoxCzAJBgNVBAYTAlJPMRAwDgYDVQQIDAdSb21hbmlhMQ8wDQYDVQQHDAZH
YWxhdGkxFTATBgNVBAoMDENsdXN0ZXItVGVzdDEQMA4GA1UEAwwHY2x1c3RlcjEf
MB0GCSqGSIb3DQEJARYQdGVjaEBjbHVzdGVyLm5ldDCCASIwDQYJKoZIhvcNAQEB
BQADggEPADCCAQoCggEBAKE8pij/D9Y2Y6qw6FlaA17Eym+FpezKcxf13s6BvVRA
QZAqQYCYjbLgbh0aTcNFowu68xyL6ozV+f4MwnNxquApzcPAyxh7W8y3AAp+oEzP
ngfjRKxHesvHpPSCLOvDeyt/soDxshvT2dJZxEeRIpRSypsg6EpBcR/oYC5gWH0G
8nMAUGwm4+t6j7YBXRYeO94++Y8N1IMffok7x22f68miTgLscl0lcsT2W7/k3J6s
lQQge2i88z7dUZ+yQHr3VbtpcMxR3hzvJE0iDomtz0UxgkdJEZ5jxp4N9+dr2XpN
NEUWG73FniH7ACeLeJRp6YCzeybnOwYbwsiDbICAK9sCAwEAAaNQME4wHQYDVR0O
BBYEFCkPxwp9co/mRUXmTOZ7qwxPIzZdMB8GA1UdIwQYMBaAFCkPxwp9co/mRUXm
TOZ7qwxPIzZdMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQELBQADggEBAAXtyODe
I6/nAFBStxMWqo4qHND/mr5T8UZMrsjgc+bRaufAStGVOLOXKrG3Q1u+7FLureOP
WgPxZmtTzZ+jWEtzGZ/weeZyjFrYLAI+7JlkPfPg8DGeYC3JXxudikmPmUvgh6lX
XmYMhl1Uah2+cxQjQRueQ8pf4pHxrlICXjYbG4PdGKGZ0duYSay24p9QsEVj+R1E
C6XRf3co8u9ej7US4P5Nxib2b1rgEX7sb3SuQRzMXzTX2pjD58PpA19APFePn8ZG
Hdw4zVHcQDrLQbarHqrjmg0dFTeG8YZEk1zIktoZutRfgTS3pAjm4KR4WXZXkbO6
JL1ZlAZrc0bbWbQ=
-----END CERTIFICATE-----
`
var rsaKeyPEM = `-----BEGIN PRIVATE KEY-----
MIIEvwIBADANBgkqhkiG9w0BAQEFAASCBKkwggSlAgEAAoIBAQChPKYo/w/WNmOq
sOhZWgNexMpvhaXsynMX9d7Ogb1UQEGQKkGAmI2y4G4dGk3DRaMLuvMci+qM1fn+
DMJzcargKc3DwMsYe1vMtwAKfqBMz54H40SsR3rLx6T0gizrw3srf7KA8bIb09nS
WcRHkSKUUsqbIOhKQXEf6GAuYFh9BvJzAFBsJuPreo+2AV0WHjvePvmPDdSDH36J
O8dtn+vJok4C7HJdJXLE9lu/5NyerJUEIHtovPM+3VGfskB691W7aXDMUd4c7yRN
Ig6Jrc9FMYJHSRGeY8aeDffna9l6TTRFFhu9xZ4h+wAni3iUaemAs3sm5zsGG8LI
g2yAgCvbAgMBAAECggEBAIf3HeNaWx931Aofjn+yoT7sUg1DNY19lxqwcofP7jk6
yk7o4z2u52bdTN3rzYGKmpiMuO/sqQjEdECgv+UcCE1p0yNs05c8h0gVVcqDx2ee
eJQMOV/SEyH+pAKwN3NE7/vZfkywBFLlDj0NB7CWGgPb4RB56oibsOqISXsyMcyT
lRsViW7Hk3KUpIlz/jgFNInEjTeLsvx5L2GHN6fXxquGEqsss4IZXQ9fCApi6xmc
xWcMklB5RN6O/6hPL4Ib4AmK8OSxOF0+odnlyDQs4k07euW5apyKpKwa/c00g0lb
cmuGvwhHIig5S5uAZ3Vzq5oPLeuh05CU0TbIqECPxVECgYEA1R1+casygkCc9Xao
bvWlQHuAJxKxo/bQ/DXNn4l2qR84ZfYsWS+GhIBN/cKZgHLllsy427DAKGY6/lmt
CDltsz5cE42eCL+wehP2dhp3iGqXFAig5af0w4me+cMExuKBOYkS7MGTKq9DqFkG
Fec1Utln0njaP4+pvCNIeepLYCkCgYEAwa6suHBp/1C2e5rWBmUf9j574ckbc6nh
9UoyDBVuooho4u9HSDb6f/ZTiQe3kjxjmpnp8M487IhW9Tzmd50+o2da3l60vIm5
M2ZNxFRgiythiqrKSCnwLRFGpFAqnEu2tZNNvJQLmmTrWFvgxSway8bxpURNg04y
6H8Bd1TZnGMCgYEAiwaHkx2MgJ4oBpVWwbPsDsnCSzsNuZnssWtj7XxedWuRfip+
udugFFYjCrTlMH9DuQFqYp7GbFRsjbrwfxn/r1ux82uCOdDbDnhxYpBXhB2M2xvZ
4peTu+/OTr7jId9nT6JVPy/0knbtWyhgKO/AwIBlE0+ViLtujfYydJ3ceCkCgYEA
srg2FPNWPAwEd0ZHHBuQRK8frRbfx/kI0kkmqVPVhREOh+l4A0EIIa/xIU8Hq18i
IfTIlDYarcCZTS5nFBT7SdkDVpJZgGgthypttC6P75uWJFi406IvR8bbQp/e0d5j
uGU2pD6P/mYFbMFLRWYPS95F+NRwGiu8eiFH/w9CxjUCgYBa01kBP/feQtJtsZ6A
59VtGzTFTpbLNTtej7uTxPOuoOAg9a9c3XnOzAL733H49mfPFvdHOFJqLuFMSiKB
XAuwYxd9hct7rW8IJQPzs/YdZIwoEGW1TEOUtUcpLDrzUiOpO8r1mj4eYeUbLGTn
peMptpBttAcUzESn1Kd5oTTT7Q==
-----END PRIVATE KEY-----
`

type TestApplication struct{}

func (c *TestApplication) OnNodeJoin(p Peer)                 {}
func (c *TestApplication) OnHeartbeat(p Peer)                {}
func (c *TestApplication) OnMessage(msg *Message)            {}
func (c *TestApplication) OnBroadcast(msg *BroadcastMessage) {}
func (c *TestApplication) OnNodeExit(p Peer)                 {}

var node1name string = "node1"
var node2name string = "node2"
var node3name string = "node3"

func TestSetLogLevel(t *testing.T) {
	cluster := &Cluster{
		logLevel: LogLevelDebug,
		optlock:  new(sync.RWMutex),
	}

	err := cluster.SetLogLevel(-1)
	if err == nil {
		t.Fatalf("Invalid Log Level accepted")
	}

	cluster.SetLogLevel(LogLevelWarn)
	if cluster.GetLogLevel() != LogLevelWarn {
		t.Fatalf("Invalid LogLevel different than we setup")
	}

}

func TestHeartbeatFrequency(t *testing.T) {

	cluster := &Cluster{
		heartbeatFrequency: 10,
		optlock:            new(sync.RWMutex),
	}

	err := cluster.SetHeartbeatFrequency(0)
	if err == nil {
		t.Fatalf("Invalid Heartbeat Frequency accepted")
	}

	err = cluster.SetHeartbeatFrequency(-10)
	if err == nil {
		t.Fatalf("Invalid Heartbeat Frequency accepted")
	}

	cluster.SetHeartbeatFrequency(2)
	if cluster.heartbeatFrequency != cluster.GetHeartbeatFrequency() {
		t.Fatalf("Invalid Heartbeat Frequency different than we setup")
	}
}

func TestID(t *testing.T) {
	// Generate new PeerId
	peer_name := "test"
	peer_addr := "127.0.0.1:10123"
	selfpeer, err := NewPeer(peer_name, "testing", peer_addr)
	if err != nil {
		t.Error(err)
	}
	cluster := &Cluster{
		self:    selfpeer,
		optlock: new(sync.RWMutex),
	}

	if cluster.StringName() != peer_name {
		t.Fatalf("Invalid peer name expected %s got %s", peer_name, cluster.StringName())
	}

	if cluster.StringAddr() != peer_addr {
		t.Fatal("Invalid peer address expected %s got %s ", peer_addr, cluster.StringAddr())
	}

	if cluster.GetID() == int64(0) {
		t.Fatal("Expected to retrieve a peer id")
	}

	if cluster.GetID() < int64(0) {
		t.Fatal("Peer id should be positive value")
	}

	if cluster.self.StringID() != fmt.Sprintf("%d", cluster.GetID()) {
		t.Fatal("Invalid peer string id")
	}

	otherpeer, err := NewPeer(peer_name, "testing", "127.0.0.1:1234")

	if selfpeer.StringAddr() == otherpeer.StringAddr() {
		t.Fatalf("Address should be differennt")
	}

	if selfpeer.equal(otherpeer) {
		t.Fatalf("Peer should be different")
	}
}

func TestSetLogger(t *testing.T) {
	cluster := &Cluster{
		heartbeatFrequency: 10,
		optlock:            new(sync.RWMutex),
	}
	logger := log.New(os.Stdout, "", log.LstdFlags)

	err := cluster.SetLogger(nil)
	if err == nil {
		t.Fatalf("Invalid logger accepted")
	}

	err = cluster.SetLogger(logger)
	if err != nil {
		t.Fatalf("Logger not accepted")
	}

}

func TestNodeJoinExit(t *testing.T) {
	var peer *Peer

	// Generate new PeerId
	selfpeer, err := NewPeer("test", "testing", "127.0.0.1:0")
	if err != nil {
		t.Error(err)
	}

	cluster := &Cluster{
		self:               selfpeer,
		heartbeatFrequency: 10,
		optlock:            new(sync.RWMutex),
		peerslock:          new(sync.RWMutex),
		peers:              make(map[int64]Peer),
		logLevel:           LogLevelDebug,
		log:                log.New(os.Stdout, "", log.LstdFlags),
	}

	peer, _ = NewPeer("test2", "local", "127.0.0.1:20")

	cluster.onNodeJoin(*peer)

	cluster.onNodeJoin(*selfpeer)

	cluster.Broadcast("test message for all")

	cluster.peers[peer.Id] = *peer
	cluster.onNodeExit(*peer, true)

	cluster.Disconnect(*peer)

}

func TestCluster(t *testing.T) {
	cert, err := tls.X509KeyPair([]byte(rsaCertPEM), []byte(rsaKeyPEM))
	if err != nil {
		t.Fatalf("loadkeys failed: %s", err)
	}
	tlsconfig := tls.Config{
		Certificates:       []tls.Certificate{cert},
		InsecureSkipVerify: true,
	}
	node1, err := New(&TestApplication{}, node1name, "local", "0.0.0.0:5001", &tlsconfig)
	if err != nil {
		t.Fatal(err)
	}

	node2, err := New(&TestApplication{}, node2name, "local", "0.0.0.0:5002", &tlsconfig)
	if err != nil {
		t.Fatal(err)
	}

	node3, err := New(&TestApplication{}, node3name, "local", "127.0.0.1:5001", nil)
	if err == nil {
		t.Fatal("Complete New() TestCase")
	}

	node3, err = New(&TestApplication{}, node3name, "local", "0.0.0.0:5003", nil)
	if err != nil {
		t.Fatal(err)
	}

	node1.Connect(node2.StringAddr())
	node2.Connect(node3.StringAddr())
	node3.Connect(node1.StringAddr())

	err = node1.Connect(node3.StringAddr())
	if err == nil {
		t.Fatalf("Should not be able to connect TLS to nonTLS")
	}

	err = node1.Connect("127.0.0.1")
	if err == nil {
		t.Fatalf("Invalid address in connect")
	}

	// Wait for peers to join
	time.Sleep(3 * time.Second)

	node1.Broadcast([]byte("TSL Broadcast"))

	// Add bogus node (100% broacast test)
	node1.peerslock.Lock()
	bogus, _ := NewPeer("test", "local", "127.0.0.1:5010")
	node1.peers[bogus.Id] = *bogus
	node1.peerslock.Unlock()
	node1.Broadcast([]byte("Bogus peeer TSL Broadcast"))

	//

	peers1 := node1.GetPeers()
	peers2 := node2.GetPeers()

	if peers1[0].Id != node2.GetID() {
		t.Fatalf("Same peer list ?")
	}

	if peers2[0].Id != node1.GetID() {
		t.Fatalf("Same peer list ?")
	}

	node1.Shutdown()
	node2.Shutdown()
}
