package cluster

import (
	"crypto/tls"
	"errors"
	"log"
	"net"
	"os"
	"strings"
	"sync"
	"time"
)

// ToDo :
//     better testing

// Cluster App Interface
type ClusterApplication interface {
	OnNodeJoin(p Peer)
	OnMessage(m *Message)
	OnBroadcast(m *BroadcastMessage)
	OnHeartbeat(p Peer)
	OnNodeExit(p Peer)
}

// Cluster class
type Cluster struct {
	tcp                net.Listener
	tls                *tls.Config
	self               *Peer
	peers              map[int64]Peer
	applications       []ClusterApplication
	heartbeats         map[int64]time.Time
	heartbeatFrequency int
	networkTimeout     int
	optlock            *sync.RWMutex
	peerslock          *sync.RWMutex
	waitgroup          *sync.WaitGroup
	message            chan *message
	receiveQuit        chan bool
	heartbeatQuit      chan bool
	logLevel           int
	log                *log.Logger
}

// Simple way to create a cluster instance
func New(app ClusterApplication, name string, domain, laddr string, tlsconfig *tls.Config) (*Cluster, error) {
	var err error

	// Generate new PeerId
	selfpeer, err := NewPeer(name, domain, laddr)
	if err != nil {
		return nil, err
	}

	// Init cluster
	cluster := &Cluster{
		tcp:                nil,
		tls:                tlsconfig,
		self:               selfpeer,
		peers:              make(map[int64]Peer),
		message:            make(chan *message),
		receiveQuit:        make(chan bool),
		heartbeats:         make(map[int64]time.Time),
		heartbeatQuit:      make(chan bool),
		heartbeatFrequency: DefaultHeartbeatFrequency,
		networkTimeout:     DefaultNetworkTimeout,
		optlock:            new(sync.RWMutex),
		peerslock:          new(sync.RWMutex),
		waitgroup:          new(sync.WaitGroup),
		logLevel:           LogLevelDebug,
		log:                log.New(os.Stdout, "", log.LstdFlags),
	}

	cluster.RegisterCallback(app)

	localaddr := cluster.self.StringAddr()

	cluster.tcp, err = net.Listen("tcp", localaddr)
	if err != nil {
		return nil, err
	}

	if cluster.tls != nil {
		cluster.tcp = tls.NewListener(cluster.tcp, cluster.tls)
		cluster.debug("Listening TLS on %s", localaddr)
	} else {
		cluster.debug("Listening on %s", localaddr)
	}

	go cluster.acceptRoutine()
	go cluster.receiveRoutine()
	go cluster.heartbeatRoutine()

	return cluster, err
}

// SetLogger sets the log.Logger that the Cluster will write to.
func (c *Cluster) SetLogger(l *log.Logger) error {
	c.optlock.Lock()
	defer c.optlock.Unlock()
	if l == nil {
		return errors.New("Invalid Logger interface")
	}
	c.log = l
	return nil
}

// SetLogLevel sets the level of logging that will be written to the Logger.
func (c *Cluster) SetLogLevel(level int) error {
	c.optlock.Lock()
	defer c.optlock.Unlock()

	if level < LogLevelDebug || level > LogLevelError {
		return errors.New("Invalid log level")
	}

	c.logLevel = level
	return nil
}

// SetHeartbeatFrequency sets the frequency in seconds with which heartbeats
//  will be sent from this Node to test the health of other Nodes in the Cluster.
func (c *Cluster) SetHeartbeatFrequency(freq int) error {
	c.optlock.Lock()
	defer c.optlock.Unlock()
	if freq < 1 {
		return errors.New("Heartbeat frequency should be positive and greater then 0 seconds")
	}
	c.heartbeatFrequency = freq
	return nil
}

// GetHeartbeatFrequency return the frequency in seconds
func (c *Cluster) GetHeartbeatFrequency() int {
	c.optlock.RLock()
	defer c.optlock.RUnlock()

	return c.heartbeatFrequency
}

// GetLogLevel return the current logLevel
func (c *Cluster) GetLogLevel() int {
	c.optlock.Lock()
	defer c.optlock.Unlock()

	return c.logLevel
}

// My node Id
func (c *Cluster) GetID() int64 {
	c.optlock.RLock()
	defer c.optlock.RUnlock()

	return c.self.Id
}

// MyNode Name as string
func (c *Cluster) StringName() string {
	c.optlock.RLock()
	defer c.optlock.RUnlock()

	return c.self.Name
}

// MyNode IP as string
func (c *Cluster) StringAddr() string {
	c.optlock.RLock()
	defer c.optlock.RUnlock()

	return c.self.StringAddr()
}

// Register Applications callback
func (c *Cluster) RegisterCallback(app ClusterApplication) {
	c.peerslock.Lock()
	defer c.peerslock.Unlock()

	c.applications = append(c.applications, app)
}

// Connect to peer by Network Address
func (c *Cluster) Connect(address string) error {
	c.peerslock.RLock()
	defer c.peerslock.RUnlock()

	addr, err := net.ResolveTCPAddr("tcp", address)
	if err != nil {
		c.err("Connect() error %s, %s", address, err)
		return err
	}

	if err := c.Send(Peer{TCPAddr: addr}, MsgJoin, c.self); err != nil {
		return err
	}

	return nil
}

// Send a message to a Peer
func (c *Cluster) Send(to Peer, Type int8, data interface{}) error {
	c.optlock.RLock()
	defer c.optlock.RUnlock()

	networkTimeout := time.Duration(c.networkTimeout) * time.Second

	raddr := to.StringAddr()
	conn, err := net.DialTimeout("tcp", raddr, networkTimeout)

	if err != nil {
		c.warn("Send: Unable to dial connection to %s (%s)", raddr, err)
		return err
	}
	defer conn.Close()

	if c.tls != nil {
		colonPos := strings.LastIndex(raddr, ":")

		if colonPos == -1 {
			colonPos = len(raddr)
		}

		hostname := raddr[:colonPos]

		// If no ServerName is set, infer the ServerName
		// from the hostname we're connecting to.
		if c.tls.ServerName == "" {
			// Make a copy to avoid polluting argument or default.
			conn := *c.tls
			conn.ServerName = hostname
			c.tls = &conn
		}
		tslconn := tls.Client(conn, c.tls)
		if err = tslconn.Handshake(); err != nil {
			conn.Close()
			c.warn("Unable to handshake TSL connection with %s", raddr)
			return err
		}
		conn = tslconn
	}

	enc := newEncoder(conn)

	deadline := time.Now().Add(networkTimeout)
	if err := conn.SetWriteDeadline(deadline); err != nil {
		return err
	}

	if err := enc.Encode(c.self.Id, to.Id, Type, data); err != nil {
		return err
	}

	return nil
}

// Broadcast a message
func (c *Cluster) Broadcast(message interface{}) {
	c.peerslock.RLock()
	defer c.peerslock.RUnlock()

	for _, peer := range c.peers {
		if err := c.Send(peer, MsgBroadcast, message); err != nil {
			c.debug("Broadcast to %s@%s failed: %s",
				peer.StringID(),
				peer.StringAddr(),
				err,
			)
		}
	}
}

// Disconnect from remote peer
func (c *Cluster) Disconnect(p Peer) {
	c.Send(p, MsgLeave, nil)
	c.onNodeExit(p, true)
}

//MySelf  Shutdown
func (c *Cluster) Shutdown() {

	// ToDo : Fix non existing cluster
	if c == nil {
		return
	}
	c.debug("Shutting down cluster ...")

	c.receiveQuit <- true
	c.heartbeatQuit <- true
	c.waitgroup.Wait()

	c.peerslock.Lock()
	defer c.peerslock.Unlock()

	c.tcp.Close()

	for _, peer := range c.peers {
		if err := c.Send(peer, MsgLeave, nil); err != nil {
			c.debug("Unable to send Goodbye to %s@%s failed: %s",
				peer.StringID(),
				peer.StringAddr(),
				err,
			)
		} else {
			// Unlearn remotes
			delete(c.peers, peer.Id)
			delete(c.heartbeats, peer.Id)
		}
	}
	c.debug("Closing DONE")

}

// Get list of peers
func (c *Cluster) GetPeers() []Peer {
	c.peerslock.RLock()
	defer c.peerslock.RUnlock()

	rs := make([]Peer, 0, len(c.peers))
	for _, r := range c.peers {
		rs = append(rs, r)
	}
	return rs
}

// Return a peer by id
func (c *Cluster) GetPeerById(peerid int64) (Peer, bool) {
	c.peerslock.RLock()
	defer c.peerslock.RUnlock()

	peer, known := c.peers[peerid]
	return peer, known
}

// on NodeJoin
func (c *Cluster) onNodeJoin(peer Peer) error {
	c.peerslock.Lock()
	defer c.peerslock.Unlock()

	if peer.equal(c.self) || peer.Id == c.self.Id {
		c.debug("Ignoring peer '%s@%s'. (Self)",
			peer.StringID(),
			peer.StringAddr(),
		)
		return nil
	}

	if _, ok := c.peers[peer.Id]; ok {
		c.debug("Ignoring peer %s@%s. (Known)",
			peer.StringID(),
			peer.StringAddr(),
		)
		return nil
	}

	// Respond to join
	if err := c.Send(peer, MsgJoin, c.self); err != nil {
		return err
	}

	c.peers[peer.Id] = peer

	// onNodeJoin notification
	for _, app := range c.applications {
		app.OnNodeJoin(peer)
	}

	// Send peer list to new remote
	var peers []Peer
	for _, r := range c.peers {
		// Send only list of peers from same domain
		if r.Domain == peer.Domain {
			peers = append(peers, r)
		}
	}
	if err := c.Send(peer, MsgPeers, peers); err != nil {
		return err
	}

	c.debug("Learned new %s@%s",
		peer.StringID(),
		peer.StringAddr(),
	)
	return nil
}

// onNodeExit
func (c *Cluster) onNodeExit(peer Peer, wipeHistory bool) {
	c.peerslock.Lock()
	defer c.peerslock.Unlock()

	if _, ok := c.peers[peer.Id]; !ok {
		return
	}

	delete(c.peers, peer.Id)

	if wipeHistory {
		delete(c.heartbeats, peer.Id)
	}

	// Send Leave notification
	for _, app := range c.applications {
		app.OnNodeExit(peer)
	}

	c.debug("Unlearned remote: %s@%s",
		peer.StringID(),
		peer.StringAddr(),
	)
}
